<?php
// app/Http/Controllers/AdminController.php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Desa;
use App\Models\MenuIcon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }
    
    public function dashboard()
    {
        $stats = [
            'total_users' => User::count(),
            'total_desa' => Desa::count(),
            'pending_users' => User::where('is_active', false)->count(),
        ];
        
        return view('admin.dashboard', compact('stats'));
    }
    
    public function users()
    {
        $users = User::with('desa', 'dusun')->paginate(20);
        $desa = Desa::all();
        
        return view('admin.users', compact('users', 'desa'));
    }
    
    public function createUser(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'username' => 'required|string|max:255|unique:users',
            'email' => 'required|email|unique:users',
            'password' => 'required|string|min:8',
            'role' => 'required|in:warga,pemdes,petugas_tangki,admin',
            'desa_id' => 'nullable|exists:desa,id',
            'phone' => 'nullable|string',
        ]);
        
        User::create([
            'name' => $validated['name'],
            'username' => $validated['username'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
            'desa_id' => $validated['desa_id'] ?? null,
            'phone' => $validated['phone'] ?? null,
            'is_active' => true,
        ]);
        
        return redirect()->back()->with('success', 'User berhasil dibuat!');
    }
    
    public function resetPassword(Request $request, $id)
    {
        $validated = $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ]);
        
        $user = User::findOrFail($id);
        $user->update([
            'password' => Hash::make($validated['password']),
        ]);
        
        return redirect()->back()->with('success', 'Password berhasil direset!');
    }
    
    public function updateUsername(Request $request, $id)
    {
        $validated = $request->validate([
            'username' => 'required|string|unique:users,username,' . $id,
        ]);
        
        $user = User::findOrFail($id);
        $user->update(['username' => $validated['username']]);
        
        return redirect()->back()->with('success', 'Username berhasil diubah!');
    }
    
    public function toggleUserStatus($id)
    {
        $user = User::findOrFail($id);
        $user->update(['is_active' => !$user->is_active]);
        
        return redirect()->back()->with('success', 'Status user berhasil diubah!');
    }
    
    public function menuIcons()
    {
        $icons = MenuIcon::orderBy('urutan')->get();
        return view('admin.menu-icons', compact('icons'));
    }
    
    public function updateMenuIcon(Request $request, $id)
    {
        $validated = $request->validate([
            'icon_class' => 'required|string',
            'urutan' => 'nullable|integer',
        ]);
        
        $menuIcon = MenuIcon::findOrFail($id);
        $menuIcon->update($validated);
        
        return redirect()->back()->with('success', 'Icon menu berhasil diubah!');
    }
}
